from django.db import models
from ckeditor.fields import RichTextField

class CategoryList(models.Model):
    name = models.CharField(max_length=100)
    slug = models.CharField(max_length=150, blank=True, null=True)
    description = models.TextField(blank=True)
    ordering = models.IntegerField(default=0) 
    image = models.ImageField(upload_to='category_images/', blank=True, null=True)  # New field
    is_homepage = models.BooleanField(default=False)
    status = models.BooleanField(default=True)
    created  = models.DateTimeField(auto_now_add=True) 
    updated  = models.DateTimeField(auto_now=False) 

    def __str__(self):
        return str(self.name)

    class Meta:
        db_table = 'category_list' 


class BrandList(models.Model):  
    brand_name          = models.CharField(max_length=60, blank=True, null=True) 
    details             = models.TextField(blank=True, null=True)
    ordering            = models.IntegerField(default=0) 
    created             = models.DateTimeField(auto_now_add=True)   
    status              = models.BooleanField(default=True)

    def __str__(self):
        return str(self.brand_name)

    class Meta:
        db_table = 'brand_list' 

class ColorList(models.Model):  
    color_name          = models.CharField(max_length=60, blank=True, null=True) 
    details             = models.TextField(blank=True, null=True)
    ordering            = models.IntegerField(default=0) 
    created             = models.DateTimeField(auto_now_add=True)   
    status              = models.BooleanField(default=True)

    def __str__(self):
        return str(self.color_name)

    class Meta:
        db_table = 'color_list' 


class SizeList(models.Model):  
    size_name           = models.CharField(max_length=60, blank=True, null=True) 
    short_code           = models.CharField(max_length=10, blank=True, null=True) 
    ordering            = models.IntegerField(default=0) 
    created             = models.DateTimeField(auto_now_add=True)   
    status              = models.BooleanField(default=True)

    def __str__(self):
        return str(self.size_name)

    class Meta:
        db_table = 'size_list' 
 
class ProductList(models.Model):     
    name_bangla              = models.CharField(max_length=250, null = True)
    name_english             = models.CharField(max_length=250, blank=True, null = True)
    slug                     = models.CharField(max_length=250, blank=True, null = True) 
    sku_code                 = models.CharField(max_length=50, blank=True)
    
    brand                   = models.ForeignKey(BrandList, on_delete=models.DO_NOTHING, null=True)  
    unit_price               = models.DecimalField(max_digits = 10, decimal_places= 2)
    discount                 = models.DecimalField(max_digits = 10, decimal_places= 2, blank=True, null = True) 
    discount_price           = models.DecimalField(max_digits = 10, decimal_places= 2, blank=True, null = True)
    discount_status          = models.IntegerField(default=0) 
    quantity                 = models.IntegerField(default=0)  
    delivery_min_day         = models.IntegerField(blank=True,null=True) 
    delivery_max_day         = models.IntegerField(blank=True,null=True) 
    is_variant               = models.BooleanField(default=False)
     
    stock = ( 
        ('1', 'Available'),
        ('2', 'Out of Stock'), 
    )
    stock_status     = models.CharField(max_length=10, choices=stock)  
    short_detail      = models.TextField(blank=True, null = True) 
    full_detail      = RichTextField(blank=True, null = True) 
    created          = models.DateField(auto_now_add=True)
    updated          = models.DateField(auto_now=False)
    deleted_status   = models.BooleanField(default=0)
    product_image    = models.ImageField(upload_to='product_img/', blank=True, null=True) 

     # ---------------- SEO FIELDS ----------------
    seo_meta_title           = models.CharField(max_length=255, blank=True, null=True)
    seo_meta_description     = RichTextField(blank=True, null=True)
    seo_focus_keyword        = models.TextField(max_length=100, blank=True, null=True)   
    seo_og_image             = models.ImageField(upload_to='product/og_images/', blank=True, null=True) 
    total_reviews = models.IntegerField(default=0)
    total_sales = models.IntegerField(default=0)
    average_rating = models.DecimalField(max_digits=3, decimal_places=2, default=0.0)
    


   
    def __str__(self):
        return self.name_english

    class Meta:
        db_table = 'product_list' 

class ProductWiseColor(models.Model):
    product         = models.ForeignKey(ProductList, on_delete=models.DO_NOTHING, null=True)
    color           = models.ForeignKey(ColorList, on_delete=models.DO_NOTHING, null=True)   
    status          = models.BooleanField(default=True)
    created         = models.DateTimeField(auto_now_add=True)
    
    def __int__(self):
        return self.product

    class Meta:
        db_table = 'product_wise_color'
        verbose_name = 'Product Wise Color'
        verbose_name_plural = 'Product Wise Colors'

class ProductWiseSize(models.Model):
    product         = models.ForeignKey(ProductList, on_delete=models.DO_NOTHING, null=True)
    size            = models.ForeignKey(SizeList, on_delete=models.DO_NOTHING, null=True) 
    unit_price      = models.DecimalField(max_digits = 10, decimal_places= 2) 
    discount        = models.DecimalField(max_digits = 10, decimal_places= 2) 
    discount_price  = models.DecimalField(max_digits = 10, decimal_places= 2) 
    quantity        = models.PositiveIntegerField(default=0)
    status          = models.BooleanField(default=True)
    created         = models.DateTimeField(auto_now_add=True)
    
    def __int__(self):
        return self.product

    class Meta:
        db_table = 'product_wise_size'
        verbose_name = 'Product Wise Size'
        verbose_name_plural = 'Product Wise Sizes'
 

class ProductWiseCategory(models.Model):
    product         = models.ForeignKey(ProductList, on_delete=models.DO_NOTHING, null=True)
    category            = models.ForeignKey(CategoryList, on_delete=models.DO_NOTHING, null=True) 
    status          = models.BooleanField(default=True)
    created         = models.DateTimeField(auto_now_add=True)
    
    def __int__(self):
        return self.product

    class Meta:
        db_table = 'product_wise_category'
        verbose_name = 'Product Wise Category'
        verbose_name_plural = 'Product Wise Categories'
 


class ProductImage(models.Model):
    product = models.ForeignKey(ProductList, on_delete=models.CASCADE)
    image = models.ImageField(upload_to='product_images/')

    class Meta:
        db_table = 'product_image'


class DistrictList(models.Model): 
    district_name_english  = models.CharField(max_length=100) 
    district_name_bangla  = models.CharField(max_length=100, blank=True, null=True) 
    rank       = models.IntegerField(default=0) 
    shipping_cost    = models.IntegerField(default = 0)  
    status         = models.BooleanField(default=True)
    created    = models.DateTimeField(auto_now_add = True)
    
    def __str__(self):
        return str(self.district_name_english)

    class Meta:
        db_table = 'district_list'

class UpozillaList(models.Model):
    district  = models.ForeignKey(DistrictList, on_delete=models.DO_NOTHING, null=True)
    name   = models.CharField(max_length=230)  
    rank       = models.IntegerField(default=0)
    status         = models.BooleanField(default=1)
    created    = models.DateTimeField(auto_now_add = True)

    def __str__(self):
        return str(self.name)
        
    class Meta:
        db_table = 'upozilla_list'


class CustomerList(models.Model):
    customer_name = models.CharField(max_length=30)
    mobile        = models.CharField(max_length=15)
    optional_mobile = models.CharField(max_length=15, blank = True)
    email         = models.CharField(max_length=170, blank=True, null = True)
    username     = models.CharField(max_length=100, blank=True)
    password      = models.CharField(max_length=100)
    login_otp     = models.IntegerField(blank=True, null = True)
    reg_date      = models.DateTimeField(auto_now_add=True)
    date_of_birth = models.DateTimeField(blank=True, null=True)
    gender      = models.CharField(max_length=10, blank=True, null=True)
    district  = models.ForeignKey(DistrictList, on_delete=models.DO_NOTHING, null=True)
    upozilla  = models.ForeignKey(UpozillaList, on_delete=models.DO_NOTHING, null=True)
    address       = models.TextField(blank=True)
    profile_images  = models.ImageField(upload_to='images/customer_images', blank = True)  
    is_guest        = models.BooleanField(default=1)
    customer_level  = models.CharField(max_length=55, blank=True, null=True) 
    total_point     = models.DecimalField(max_digits=30, decimal_places=6, default=0)
    used_point      = models.DecimalField(max_digits=30, decimal_places=6, default=0)
    available_point = models.DecimalField(max_digits=30, decimal_places=6, default=0)
    created         = models.DateTimeField(auto_now_add=True, null=True)
    status          = models.BooleanField(default=1)

    def __str__(self):
        return self.customer_name

    class Meta:
        db_table = 'customer_list'
    


class CourierService(models.Model): 
    courier_name    = models.CharField(max_length=60, blank=True, null = True)
    courier_address = models.TextField(blank=True, null = True)
    ordering        = models.IntegerField(default=0)
    status          = models.BooleanField(default=1) 

    def __str__(self):
        return self.courier_name

    class Meta:
        db_table = 'courier_service_list'


class SalesOrder(models.Model): 
    order_number     = models.IntegerField(default=0)
    customer    = models.ForeignKey(CustomerList, on_delete=models.DO_NOTHING, blank = True, null=True) 
    shipping_person_name    = models.CharField(max_length=180)
    shipping_mobile = models.CharField(max_length=15) 
    shipping_email  = models.EmailField(max_length=150, blank = True) 
    shipping_address  = models.CharField(max_length=20000) 
    total_amount     = models.IntegerField(default=0)
    vat_amount     = models.IntegerField(default=0)
    discount_amount     = models.IntegerField(default=0)
    less_amount      = models.IntegerField(default=0)
    due_amount      = models.IntegerField(default=0) 
    shipping_charge  = models.IntegerField(default=0)
    service_charge   = models.IntegerField(default=0)
    service_code   = models.IntegerField(default=0)
    grand_total      = models.IntegerField(default=0)
    payment_type_choose = (
        ('cod', 'COD'),
        ('bkash', 'Bkash'),
        ('rocket', 'Rocket'),
        ('nagad', 'Nagad'),
        ('aamarpay', 'AamarPay'),
    )
    payment_method  = models.CharField(max_length=20, choices=payment_type_choose, default=1) 
    delivery_method = models.ForeignKey(CourierService, on_delete=models.DO_NOTHING, blank = True, null=True)
    payment_status_choose = (
        ('pending', 'Pending'), 
        ('paid', 'Paid'),
        ('refund', 'Refund'),
    )
    payment_status  = models.CharField(max_length=20, choices=payment_status_choose, default=1)
    order_status_choose = (
        ('1', 'Pending'),
        ('2', 'Confirmed'),
        ('3', 'Packed'),
        ('4', 'Shipping'),
        ('5', 'Delivered'),
        ('6', 'Returned'),
        ('7', 'Canceled'),
        ('8', 'Hold'),
    )
    order_status  = models.CharField(max_length=20, choices=order_status_choose, default=1) 
    district  = models.ForeignKey(DistrictList, on_delete=models.DO_NOTHING, null=True)
    upozilla  = models.ForeignKey(UpozillaList, on_delete=models.DO_NOTHING, null=True)
    session_key      = models.CharField(max_length=230)
    order_date       = models.DateTimeField(auto_now_add=True)
    updated      = models.DateTimeField(auto_now_add=False, null = True)
    order_remarks    = models.TextField(blank=True,null=True) 

    def __int__(self):
        return self.order_number

    class Meta:
        db_table = 'sales_order'
        verbose_name = 'Sales Order'
        verbose_name_plural = 'Sales Orders'


class SalesOrderDetails(models.Model):  
    order_no         = models.ForeignKey(SalesOrder, on_delete=models.DO_NOTHING, blank = True, null=True) 
    product         = models.ForeignKey(ProductList, on_delete=models.DO_NOTHING, blank = True, null=True) 
    unit_price               = models.DecimalField(max_digits = 10, decimal_places= 2)
    discount                 = models.DecimalField(max_digits = 10, decimal_places= 2, blank=True, null = True) 
    discount_price           = models.DecimalField(max_digits = 10, decimal_places= 2, blank=True, null = True) 
    quantity   = models.IntegerField(default=1) 
    total_price           = models.DecimalField(max_digits = 10, decimal_places= 2, blank=True, null = True)   
    remarks         = models.TextField(blank = True)
    
    color = models.ForeignKey(ColorList, on_delete=models.CASCADE, null=True, blank=True)
    size = models.ForeignKey(SizeList, on_delete=models.CASCADE, null=True, blank=True)
    
    created      = models.DateTimeField(auto_now_add=True)
    updated     = models.DateTimeField(auto_now_add=False, blank = True, null=True) 
    status          = models.BooleanField(default=1) 
    def __int__(self):
        return self.order_number

    class Meta:
        db_table = 'sales_order_details'
        verbose_name = 'Sales Order Detail'
        verbose_name_plural = 'Sales Order Details'
 



class ProductReview(models.Model):
    product = models.ForeignKey(ProductList, on_delete=models.CASCADE)
    name = models.CharField(max_length=100)
    email = models.EmailField()
    rating = models.IntegerField()
    review = models.TextField()
    created = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return f"{self.name} - {self.rating}"
    
    class Meta:
        db_table = 'product_review'
        verbose_name = 'Product Review'
        verbose_name_plural = 'Product Reviews'


# Recent View Product List 
class RecentViewProduct(models.Model):
    product = models.ForeignKey(ProductList, on_delete=models.CASCADE)
    session_key = models.CharField(max_length=250)
    view_date = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return f"{self.session_key} - {self.product.name_english}"

    class Meta:
        db_table = 'recent_view_products'
        verbose_name = 'Recent View Product'
        verbose_name_plural = 'Recent View Products'
