from django.db import models 
from inventory_app.models import * 



class BusinessProfile(models.Model):
    # Basic Information
    business_name       = models.CharField(max_length=250)
    tagline             = models.CharField(max_length=250, blank=True, null=True)
    description         = models.TextField(blank=True, null=True)

    # Contact Information
    email               = models.EmailField(max_length=150, blank=True, null=True)
    mobile               = models.CharField(max_length=20, blank=True, null=True)
    mobile_optional      = models.CharField(max_length=20, blank=True, null=True)
    address             = models.TextField(blank=True, null=True)
    map_embed_code      = models.TextField(blank=True, null=True) 
    logo                = models.ImageField(upload_to="business/logo/", blank=True, null=True)
    favicon             = models.ImageField(upload_to="business/favicon/", blank=True, null=True) 

    # shipping_Cost 
    shipping_cost_inside = models.IntegerField(default=0) 
    shipping_cost_outside = models.IntegerField(default=0) 

    # Social Media
    facebook            = models.URLField(max_length=250, blank=True, null=True)
    instagram           = models.URLField(max_length=250, blank=True, null=True)
    twitter             = models.URLField(max_length=250, blank=True, null=True)
    linkedin            = models.URLField(max_length=250, blank=True, null=True)
    youtube             = models.URLField(max_length=250, blank=True, null=True)

    # SEO
    seo_meta_title      = models.CharField(max_length=70, blank=True, null=True)
    seo_meta_description= models.CharField(max_length=160, blank=True, null=True)
    seo_keywords        = models.CharField(max_length=250, blank=True, null=True)

    # Settings
    is_active           = models.BooleanField(default=True)
    created             = models.DateTimeField(auto_now_add=True)
    updated             = models.DateTimeField(auto_now=True)

    def __str__(self):
        return self.business_name

    class Meta:
        db_table = "business_profile"
        verbose_name = "Business Profile"
        verbose_name_plural = "Business Profiles"


class SliderList(models.Model):
    title       = models.CharField(max_length=200, blank=True, null=True)
    subtitle    = models.CharField(max_length=250, blank=True, null=True) 
    image       = models.ImageField(upload_to="sliders/") 
    ordering    = models.PositiveIntegerField(default=0, help_text="Order of the slider")
    status      = models.BooleanField(default=True)
    created     = models.DateTimeField(auto_now_add=True)
    updated     = models.DateTimeField(auto_now=True)

    class Meta:
        db_table = "slider_list" 
        verbose_name = "Slider"
        verbose_name_plural = "Sliders"

    def __str__(self):
        return self.title if self.title else f"Slider {self.id}"



class AddToCart(models.Model):
    session_key = models.CharField(max_length=50, null=True, blank=True)
    customer = models.ForeignKey(CustomerList, on_delete=models.CASCADE, null=True, blank=True)
    product = models.ForeignKey(ProductList, on_delete=models.CASCADE)
    quantity = models.PositiveIntegerField(default=1)
    color = models.ForeignKey(ColorList, on_delete=models.CASCADE, null=True, blank=True)
    size = models.ForeignKey(SizeList, on_delete=models.CASCADE, null=True, blank=True)

    # Price fields
    unit_price = models.DecimalField(max_digits=12, decimal_places=2, default=0.00)
    discount_price = models.DecimalField(max_digits=12, decimal_places=2, default=0.00)
    discount = models.DecimalField(max_digits=12, decimal_places=2, default=0.00)

    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    
    class Meta:
        db_table = "add_to_cart"  
        verbose_name = "Add To Cart"
        verbose_name_plural = "Add To Carts"

        unique_together = ('session_key', 'product')
 

 
class Department(models.Model):
    name = models.CharField(max_length=100, unique=True)
    rank = models.IntegerField() 
    status = models.BooleanField(default=True)
    created = models.DateTimeField(auto_now_add=True)

    class Meta:
        db_table = 'department_list'

    def __str__(self):
        return self.name


class Designation(models.Model):
    name = models.CharField(max_length=100, unique=True)
    rank = models.IntegerField() 
    status = models.BooleanField(default=True)
    created = models.DateTimeField(auto_now_add=True)

    class Meta:
        db_table = 'designation_list'

    def __str__(self):
        return self.name



class Module(models.Model):
    name = models.CharField(max_length=100, unique=True)
    rank = models.IntegerField(default=1)
    icon = models.CharField(max_length=100, blank=True, null=True)
    color = models.CharField(max_length=100, blank=True, null=True)
    status = models.BooleanField(default=True)
    created = models.DateTimeField(auto_now_add=True)

    class Meta:
        db_table = 'module_list'

    def __str__(self):
        return self.name

class ModuleMenu(models.Model):
    module = models.ForeignKey(Module, on_delete=models.SET_NULL, null=True, related_name='menus')
    name = models.CharField(max_length=100, unique=True)
    rank = models.IntegerField(default=1)
    menu_url = models.CharField(max_length=100, blank=True, null=True)
    icon = models.CharField(max_length=100, blank=True, null=True)
    color = models.CharField(max_length=100, blank=True, null=True)
    status = models.BooleanField(default=True)
    created = models.DateTimeField(auto_now_add=True)

    class Meta:
        db_table = 'module_menu_list'

    def __str__(self):
        return self.name

 

class Employee(models.Model):
    employee_id = models.CharField(max_length=20, unique=True)
    password = models.CharField(max_length=128)
    full_name = models.CharField(max_length=150)
    email = models.EmailField(unique=True)
    phone = models.CharField(max_length=20, blank=True, null=True)
    address = models.TextField(blank=True, null=True)
    department = models.ForeignKey(Department, on_delete=models.SET_NULL, null=True, related_name='employees')
    designation = models.ForeignKey(Designation, on_delete=models.SET_NULL, null=True, related_name='employees')
    join_date = models.DateField()
    salary_basic = models.DecimalField(max_digits=10, decimal_places=2)
    salary_gross = models.DecimalField(max_digits=10, decimal_places=2)
    national_id = models.CharField(max_length=30, blank=True, null=True)
    photo = models.ImageField(upload_to='employee_photos/', blank=True, null=True)
    is_active = models.BooleanField(default=True)
    is_delete = models.BooleanField(default=False)
    created = models.DateTimeField(auto_now_add=True)
    updated = models.DateTimeField(auto_now=True)

    class Meta:
        db_table = 'employee_list' 

    def __str__(self):
        return f"{self.employee_id} - {self.full_name}"
    

class UserAccessControl(models.Model):
    module = models.ForeignKey(Module, on_delete=models.SET_NULL, null=True, related_name='access_controls')
    menu = models.ForeignKey(ModuleMenu, on_delete=models.SET_NULL, null=True, related_name='access_controls')
    employee = models.ForeignKey(Employee, on_delete=models.SET_NULL, null=True, related_name='accesses')
    created_by = models.ForeignKey(Employee, on_delete=models.SET_NULL, null=True, related_name='created_accesses')

    can_create = models.BooleanField(default=False)
    can_read = models.BooleanField(default=False)
    can_update = models.BooleanField(default=False)
    can_delete = models.BooleanField(default=False)

    class Meta:
        db_table = 'user_access_control'

    def __str__(self):
        return f"AccessControl for {self.employee} on {self.menu}"


class Attendance(models.Model):
    employee = models.ForeignKey(Employee, on_delete=models.CASCADE)
    date = models.DateField()
    status = models.CharField(max_length=10, choices=[('Present', 'Present'), ('Absent', 'Absent')])

    class Meta:
        db_table = 'attendance'

class Payroll(models.Model):
    employee = models.ForeignKey(Employee, on_delete=models.CASCADE)
    month = models.CharField(max_length=20)
    salary = models.DecimalField(max_digits=10, decimal_places=2)
    paid_on = models.DateField()

    class Meta:
        db_table = 'payroll'

class LeaveRequest(models.Model):
    employee = models.ForeignKey(Employee, on_delete=models.CASCADE)
    from_date = models.DateField()
    to_date = models.DateField()
    reason = models.TextField()
    status = models.CharField(max_length=20, choices=[('Pending', 'Pending'), ('Approved', 'Approved'), ('Rejected', 'Rejected')], default='Pending')

    class Meta:
        db_table = 'leave_request'

 
