from django.shortcuts import render, get_object_or_404, redirect 
from inventory_app.models import * 
from hr.models import * 
from django.http import JsonResponse 
import requests
from django.db.models import Count, Q
import json
from django.views.decorators.csrf import csrf_exempt
from django.contrib import messages 
from django.db import transaction
from datetime import datetime
from django.utils import timezone

def home(request): 
    slider_list = SliderList.objects.filter(status=True).order_by('ordering')
    categories = CategoryList.objects.filter(status=True).order_by('ordering')

    product_list = ProductList.objects.all()

    session_key = request.session.session_key
    if not session_key:
        request.session.create()
        session_key = request.session.session_key


    
    # check if offer has been shown today
    last_offer_time = request.session.get('offer_popup_time')
    show_offer = True
    today = timezone.now().date()

    if last_offer_time:
        last_offer_date = timezone.datetime.fromisoformat(last_offer_time).date()
        if last_offer_date == today:
            show_offer = False


    homepage_categories = CategoryList.objects.filter(is_homepage=True, status=True).order_by('ordering')

    recent_viewed_products = RecentViewProduct.objects.filter(session_key=session_key).select_related('product')[:10]

    
    category_products = []
    for cat in homepage_categories:
        # এই category এর সাথে সম্পর্কিত সর্বশেষ 10 products
        products = (
            ProductList.objects.filter(
                productwisecategory__category=cat,
                deleted_status=False
            )
            .order_by('-id')
            .distinct()[:10]
        )
        
        if products.exists():
            category_products.append({
                'category': cat,
                'products': products
            })



    context = {
        'slider_list':slider_list,
        'categories':categories,
        'product_list':product_list,
        'category_products':category_products,
        'recent_viewed_products': recent_viewed_products,
        'show_offer': show_offer,
    }
    return render(request, 'website/home/index.html', context)



@csrf_exempt
def offer_shown(request):
    if request.method == "POST":
        data = json.loads(request.body)
        if data.get("shown"):
            request.session['offer_popup_time'] = timezone.now().isoformat()
        return JsonResponse({"status":"ok"})

        


def product_detail(request, product_id, slug):
    # Get main product
    product = get_object_or_404(ProductList, id=product_id)
    reviews = ProductReview.objects.filter(product=product).order_by('-created')

    categories = ProductWiseCategory.objects.filter(product_id = product_id).values('category')
    related_products = ProductList.objects.filter(productwisecategory__category__in=categories).exclude(id=product_id).distinct()[:8]

    product_images = ProductImage.objects.filter(product_id = product_id)
    product_colors = ProductWiseColor.objects.filter(product_id = product_id)
    product_sizes = ProductWiseSize.objects.filter(product_id = product_id)
    saved_amount = product.unit_price - product.discount_price if product.discount_price else 0


 
    session_key = request.session.session_key

    if not session_key:
        request.session.create()
        session_key = request.session.session_key

    # Check if already exists, then update time
    obj, created = RecentViewProduct.objects.update_or_create(
        session_key=session_key,
        product=product,
        defaults={'view_date': timezone.now()}
    )
 

    context = {
        "product": product,
        "product_images": product_images,
        "product_sizes": product_sizes,
        "product_colors": product_colors,
        "saved_amount": saved_amount,
        "reviews": reviews,
        "related_products": related_products,
        # "recommended_products": recommended_products,
    }
    return render(request, "website/product_detail.html", context)



def add_to_cart_ajax(request):
    if request.method == "POST":
        product_id = request.POST.get("product_id")
        quantity = int(request.POST.get("quantity", 1))
        color_id = request.POST.get("color_id") 
        size_id = request.POST.get("size_id")
        
        pro_color_id = None
        pro_size_id  = None
        if color_id:
            pro_color_id =  color_id
        
        if size_id:
            pro_size_id = size_id

        product = get_object_or_404(ProductList, id=product_id)

        # session_key create
        session_key = request.session.session_key
        if not session_key:
            request.session.create()
            session_key = request.session.session_key

        # Product prices
        unit_price = product.unit_price
        discount_price = product.discount_price
        discount = product.discount

        cart_item, created = AddToCart.objects.get_or_create(
            session_key=session_key,
            product=product,
            defaults={
                'quantity': quantity,
                'unit_price': unit_price,
                'discount_price': discount_price,
                'discount': discount,
                'color_id': pro_color_id,
                'size_id': pro_size_id,
            }
        )

        if not created:
            cart_item.quantity += quantity
            cart_item.unit_price = unit_price
            cart_item.discount_price = discount_price
            cart_item.discount = discount
            cart_item.save()

        
        cart_count = AddToCart.objects.filter(session_key=session_key).count()

        return JsonResponse({"status": "success", 'cart_count': cart_count, "message": "Product added to cart"})

    return JsonResponse({"status": "error", "message": "Invalid request"})


def cart_list(request):
    # session key handle
    if not request.session.session_key:
        request.session.create()
    session_key = request.session.session_key

     
    cart_items = AddToCart.objects.filter(session_key=session_key)

    # subtotal 
    subtotal = 0
    shipping = 100  

    for item in cart_items: 
        price = item.product.discount_price if item.product.discount_price else item.product.unit_price
        item.subtotal = price * item.quantity
        subtotal += item.subtotal

    total = subtotal + shipping

    context = {
        'cart_items': cart_items,
        'subtotal': subtotal,
        'shipping': shipping,
        'total': total,
    }
    return render(request, 'website/cart_list.html', context)


def remove_cart_item(request, item_id):
    if request.method == "POST":
        try:
            cart_item = AddToCart.objects.get(id=item_id)
            cart_item.delete()
  
            return JsonResponse({
                "status": "success",
                "message": "Item removed successfully!",
               
            })
        except AddToCart.DoesNotExist:
            return JsonResponse({"status": "error", "message": "Item not found."})
    return JsonResponse({"status": "error", "message": "Invalid request."})



def update_cart_quantity(request, item_id):
    if request.method == "POST":
        action = request.POST.get("action")
        try:
            session_key = request.session.session_key
            cart_item = AddToCart.objects.get(id=item_id, session_key=session_key)
            
            if action == "increase":
                cart_item.quantity += 1
            elif action == "decrease" and cart_item.quantity > 1:
                cart_item.quantity -= 1

            cart_item.save()
 

            return JsonResponse({
                "status": "success",
                "quantity": cart_item.quantity,
                
            })
        except AddToCart.DoesNotExist:
            return JsonResponse({"status": "error", "message": "Item not found."})

    return JsonResponse({"status": "error", "message": "Invalid request."})




# def checkout(request):

#     if not request.session.session_key:
#         request.session.create()
#     session_key = request.session.session_key

     
#     cart_items = AddToCart.objects.filter(session_key=session_key)

#     # subtotal 
#     subtotal = 0
#     shipping = 100  

#     for item in cart_items: 
#         price = item.product.discount_price if item.product.discount_price else item.product.unit_price
#         item.subtotal = price * item.quantity
#         subtotal += item.subtotal

#     total = subtotal + shipping

#     context = {
#         'cart_items': cart_items,
#         'subtotal': subtotal,
#         'shipping': shipping,
#         'total': total,
#     }
     
#     return render(request, 'website/checkout.html', context)




def checkout(request):
    if not request.session.session_key:
        request.session.create()
    session_key = request.session.session_key

    cart_items = AddToCart.objects.filter(session_key=session_key)

    # subtotal calculation
    subtotal = 0
    for item in cart_items:
        price = item.product.discount_price if item.product.discount_price else item.product.unit_price
        item.subtotal = price * item.quantity
        subtotal += item.subtotal

    shipping = 100  # default shipping
    discount_amount = 0  # later can be set from coupon
    grand_total = subtotal - discount_amount + shipping

    # ------------- SAVE ORDER ON POST -------------
    if request.method == "POST":
        try:
            with transaction.atomic():
                # collect form data
                full_name = request.POST.get("full_name")
                phone = request.POST.get("phone")
                email = request.POST.get("email")
                address = request.POST.get("address")
                shipping_method = request.POST.get("shipping_method")  # numeric (inside/outside Dhaka)
                payment_method = request.POST.get("payment_method")    # cod/bkash/nagad
                coupon_code = request.POST.get("coupon_code", None)

                print("full_name: ", full_name)
                print("phone: ", phone)
                print("email: ", email)
                print("address: ", address)
                print("full_name: ", full_name)
                print("shipping_method: ", shipping_method)
                print("payment_method: ", payment_method)
                print("coupon_code: ", coupon_code)

                # overwrite shipping from selected radio button
                shipping = int(float(shipping_method)) if shipping_method else 0
                grand_total = subtotal - discount_amount + shipping

                
                print("grand_total: ", grand_total)

                # Generate order number (simple auto-increment)
                last_order = SalesOrder.objects.order_by("-id").first()
                new_order_number = last_order.order_number + 1 if last_order else 1001



                chk_customer = CustomerList.objects.filter(mobile = phone).first()
                customer = None
                if not chk_customer:
                    customer = CustomerList.objects.create(
                        customer_name = full_name,
                        mobile = phone, email=email,
                        username = phone, password = phone,
                        address = address
                    )


                

                # save SalesOrder
                sales_order = SalesOrder.objects.create(
                    order_number=new_order_number,
                    customer=customer,  # optional, set if logged in
                    shipping_person_name=full_name,
                    shipping_mobile=phone,
                    shipping_email=email,
                    shipping_address=address,
                    total_amount=subtotal,
                    vat_amount=0,
                    discount_amount=discount_amount,
                    less_amount=0,
                    due_amount=0,
                    shipping_charge=shipping,
                    service_charge=0,
                    service_code=0,
                    grand_total=grand_total,
                    payment_method=payment_method,
                    delivery_method=None,  # later: assign courier
                    payment_status="pending",
                    order_status="1",  # Pending
                    district=None,
                    upozilla=None,
                    session_key=session_key,
                    order_date=datetime.now(),
                    order_remarks=f"Coupon: {coupon_code}" if coupon_code else None,
                )

                # save SalesOrderDetails
                for item in cart_items:
                    price = item.product.discount_price if item.product.discount_price else item.product.unit_price
                 
                    SalesOrderDetails.objects.create(
                        order_no=sales_order,
                        product=item.product,
                        unit_price=price,
                        discount=item.product.discount if hasattr(item.product, "discount") else 0,
                        discount_price=item.product.discount_price if item.product.discount_price else None,
                        quantity=item.quantity,
                        total_price=item.subtotal,
                        color_id=item.color_id,
                        size_id=item.size_id,
                        remarks="",
                    )

                    get_product = ProductList.objects.get(id=item.product.id)
                    get_product.total_sales += item.quantity
                    get_product.save()

                # clear cart after order placed
                cart_items.delete()

                messages.success(request, "✅ Order placed successfully!")
                return redirect("order_success", order_id=sales_order.id)

        except Exception as e:
            messages.error(request, f"Something went wrong: {str(e)}")

    # ----------------------------------------------

    context = {
        "cart_items": cart_items,
        "subtotal": subtotal,
        "shipping": shipping,
        "total": grand_total,
    }
    return render(request, "website/checkout.html", context)




def order_success(request, order_id):
    order = get_object_or_404(SalesOrder, id=order_id)
    order_items = SalesOrderDetails.objects.filter(order_no=order)

    context = {
        "order": order,
        "order_items": order_items,
    }
    return render(request, "website/order_success.html", context)



def products_list(request, category_id=None):
    # Fetch all active categories with product counts
    categories = CategoryList.objects.filter(status=True).order_by('ordering').annotate(
        product_count=Count('productwisecategory')
    )

    # If a category_id is provided, show products for that category
    if category_id:
        # Get product IDs linked to the category
        product_ids = ProductWiseCategory.objects.filter(category_id=category_id).values_list('product_id', flat=True)
        product_list = ProductList.objects.filter(id__in=product_ids)
    else:
        # Otherwise show all products
        product_list = ProductList.objects.all()

    context = {
        "categories": categories,
        "product_list": product_list,
        "selected_category": category_id,  # optional, for template highlight
    }

    return render(request, 'website/products.html', context)




@csrf_exempt
def filter_products(request):
    if request.method == "POST":
        data = json.loads(request.body)
        categories = data.get("categories", [])
        price = float(data.get("price", 10000))
        sort = data.get("sort", "")

        products = ProductList.objects.filter(discount_price__lte=price)

        if categories:
            products = products.filter(
                id__in=ProductWiseCategory.objects.filter(category_id__in=categories).values_list('product_id', flat=True)
            )
        # Sorting
        if sort == "low":
            products = products.order_by("discount_price")
        elif sort == "high":
            products = products.order_by("-discount_price")

        product_list = []
        for prod in products:
            product_list.append({
                "id": prod.id,
                "name": prod.name_english,
                "price": prod.discount_price,
                "old_price": prod.unit_price if prod.discount_price < prod.unit_price else None,
                "image": prod.product_image.url if prod.product_image else "/static/img/no-image.png",
                "stock_status": prod.stock_status,
                "slug": prod.slug,
            })

        return JsonResponse({"products": product_list})

    return JsonResponse({"error": "Invalid request"}, status=400)



def search_products(request):
    query = request.GET.get("q", "")
    products = []
    if query:
        qs = ProductList.objects.filter(
            Q(name_english__icontains=query) | Q(sku_code__icontains=query)
        )[:10]  # সর্বোচ্চ ১০টা product show করবে
        for p in qs:
            products.append({
                "id": p.id,
                "name": p.name_english,
                "sku": p.sku_code,
                "slug": p.slug,
                "image": p.product_image.url if p.product_image else "",
            })
    return JsonResponse({"products": products})

def aboutus(request):
    return render(request, "website/home/about.html")


def contact_us(request):
    return render(request, "website/home/contact.html")




def get_size_price(request):
    product_id = request.POST.get("product_id")
    size_id = request.POST.get("size_id")

    if not product_id or not size_id:
        return JsonResponse({"status": "error", "message": "Product or Size missing"})

    product = get_object_or_404(ProductList, id=product_id)
    size_obj = get_object_or_404(ProductWiseSize, size_id=size_id, product_id=product_id)

    discount_price = size_obj.discount_price or product.discount_price
    unit_price = size_obj.unit_price or product.unit_price

    saved_amount = unit_price - discount_price if discount_price else 0
    discount = round((saved_amount / unit_price) * 100, 2) if saved_amount > 0 else 0
  
    return JsonResponse({
        "status": "success",
        "discount_price": discount_price,
        "unit_price": unit_price,
        "discount": discount,
        "saved_amount": saved_amount,
    })



def order_tracking(request):
    order = None
    order_items = None
    order_number = None
    error_message = None

    if request.method == "POST":
        order_number = request.POST.get("order_number")
        try:
            order = SalesOrder.objects.get(order_number=order_number)
            order_items = SalesOrderDetails.objects.filter(order_no=order)
        except SalesOrder.DoesNotExist:
            error_message = "⚠️ Sorry, no order found with this number."

    context = {
        "order": order,
        "order_items": order_items,
        "order_number": order_number,
        "error_message": error_message,
    }
    return render(request, "website/home/order_tracking.html", context)




def customer_login(request):

    if request.method=="POST":
        username = request.POST.get("username")
        password = request.POST.get("password")

        chk_data = CustomerList.objects.filter(username=username, password = password).first()

        if chk_data:
            request.session['customer_id'] = chk_data.id
            request.session['customer_name'] = chk_data.customer_name
            request.session['customer_email'] = chk_data.email
            request.session['customer_phone'] = chk_data.mobile
            request.session['customer_status'] = chk_data.status

            return redirect('/customer/dashboard/')
 
    return render(request, "website/home/customer_login.html")


def customer_logout(request):
    request.session['customer_id'] = False

    return redirect('/')
    

 
def customer_register(request):
    if request.method == "POST":
        name = request.POST.get("customer_name")
        mobile = request.POST.get("mobile")
        email = request.POST.get("email")
        username = request.POST.get("username") 
        password = request.POST.get("confirm_password")
  
        # Check if mobile already exists
        if CustomerList.objects.filter(mobile=mobile).exists():
            messages.error(request, "Customer already registered!")
            return redirect("customer_register")

        # Save customer
        customer = CustomerList(
            customer_name=name,
            mobile=mobile,
            email=email,
            username=username,
            password=password  
        )
        customer.save()

        messages.success(request, "Account created successfully! Please login.")
        return redirect("customer_login")

    
    return render(request, "website/home/customer_register.html")

 
def customer_dashboard(request):
     
    customer_id = request.session.get("customer_id")
    if not customer_id:
        return redirect("customer_login")

    customer = CustomerList.objects.get(id=customer_id)
    
    return render(request, "website/customer/customer_dashboard.html", {"customer": customer})

 
def customer_profile_update(request):
     
    customer_id = request.session.get("customer_id")
    if not customer_id:
        return redirect("customer_login")

    customer = CustomerList.objects.get(id=customer_id)

    districts = DistrictList.objects.all().order_by('rank')


    if request.method == "POST": 
        customer.customer_name = request.POST.get("customer_name", customer.customer_name)
        customer.email = request.POST.get("email", customer.email)
        customer.mobile = request.POST.get("mobile", customer.mobile)
        customer.date_of_birth = request.POST.get("date_of_birth", customer.date_of_birth)
        customer.gender = request.POST.get("gender", customer.gender)
        customer.address = request.POST.get("address", customer.address)
        customer.district_id = int(request.POST.get("district", customer.district_id)) 

        # Handle profile photo upload
        if request.FILES.get("photo"):
            customer.profile_images = request.FILES["photo"]

        # Save updates
        customer.save()
        messages.success(request, "Your profile has been updated successfully!")
        return redirect("/customer/dashboard/") 
    

    
    return render(request, "website/customer/profile_update.html", {"customer": customer, "districts": districts})




from django.views.decorators.csrf import csrf_exempt
from django.utils.timesince import timesince

@csrf_exempt
def save_review(request, product_id):
    if request.method == "POST":
        name = request.POST.get("name")
        email = request.POST.get("email")
        rating = request.POST.get("rating")
        review_text = request.POST.get("review")

        product = get_object_or_404(ProductList, id=product_id)

        review = ProductReview.objects.create(
            product=product,
            name=name,
            email=email,
            rating=rating,
            review=review_text
        )

        product.total_reviews = ProductReview.objects.filter(product=product).count() 
        product.save()


        return JsonResponse({
            "status": "success",
            "name": review.name,
            "rating": int(review.rating),
            "review": review.review,
            "created": timesince(review.created) + " ago"
        })

    return JsonResponse({"status": "error", "message": "Invalid request"})
 